/**
 * 加班
 */
import moment = require("moment");
import { APPROVETYPE, ROLEENUM, STATED, SYSTEMCONSTCONFIGENUM, TABLENAMEENUM } from "../config/enum";
import { selectDataListByParam, selectDataToTableAssociation, selectOneDataByParam } from "../model/findData";
import { BizError } from "../util/bizError";
import { ERRORENUM } from "../config/errorEnum";
import { queue } from "./system";
import { getOvertimeId } from "../util/myTools";
import { addData } from "../model/addData";
import { findUserAskCount } from "./member";
import { updateManyData } from "../model/updateData";
import { delData } from "../model/delData";


/**
 * 填报加班
 * @param userId 填报人
 * @param time 工时
 * @param projectId 项目号
 * @param date 填报时间
 * @param workLog 工作内容
 * @returns {isSuccess:true, message:"添加成功"}
 */
export async function addOvertimeLog(userId:number, time:number, projectId:string, date:string, workLog:string ) {
    let dateKey = moment(date).format("YYYY-MM-DD")+ ' 18:00:00'

    //查找项目编号
    let projectInfo = await selectOneDataByParam(TABLENAMEENUM.项目表, {projectId}, []);
    if ( !projectInfo ) throw new BizError(ERRORENUM.不存在该项目, `${userId} 填报 ${projectId}`);

    //获取个人信息
    let userInfo = await selectOneDataByParam(TABLENAMEENUM.用户表, {userId}, []);
    let approver = 0; 
    if (userInfo.roleId == ROLEENUM.员工) {
        approver = queue[SYSTEMCONSTCONFIGENUM.部门经理][userInfo.departmentId].userId;
    } else if (userInfo.roleId == ROLEENUM.部门经理) {
        approver = queue[SYSTEMCONSTCONFIGENUM.加班默认].userId;
    }

    let id = getOvertimeId(userId);

    let addInfo = {id, projectId, userId, time, state:STATED.否, dateKey, workLog, approver};
    await addData(TABLENAMEENUM.加班, addInfo);

    return {isSuccess:true, message:"添加成功"};
}


/**
 * 加班列表
 * @param userId 填报人
 * @returns { 本周填报的工时信息, 本月工时统计 }
 */
export async function overtimeList(userId:number) {
    let includeConf = {};
    includeConf[TABLENAMEENUM.项目表] = ["name", "projectId"];
    let column = ["id", "userId", "time", "dateKey", "workLog", "state", "projectId"];

    let overtimeLog = await selectDataToTableAssociation(TABLENAMEENUM.加班, includeConf, {userId}, column);

    overtimeLog.sort((a, b) => {return new Date(b.dateKey).valueOf() - new Date(a.dateKey).valueOf() });
    let notApprove = 0;
    let dataList = [];
    let overtimeCount = 0;
    let approveCount = 0; 
    overtimeLog.forEach(info => {
        let {id, projectId, userId, time, dateKey, workLog, state, project } = info;
        notApprove += state == 0 ? time : 0;
        approveCount += state ==  APPROVETYPE.通过 ? time : 0;
        overtimeCount += time;
        let stateStr = "未审批"
        if (state) stateStr = state == APPROVETYPE.通过 ? "通过" : "未通过";
        dataList.push({
            dateKey:moment(dateKey).format("YYYY-MM-DD"),
            id, projectId, userId, time, workLog,
            projectName:project.name,
            state: stateStr
        });
       
    });
    let {cto} = await findUserAskCount(userId);
    
    return { dataList, notApprove, available:approveCount-cto, overtimeCount };
}


/**
 * 修改加班
 * @param id 
 * @param userId 填报人
 * @param time 工时
 * @param projectId 项目号
 * @param date 填报时间
 * @param workLog 工作内容
 * @returns {isSuccess:true, message:"添加成功"}
 */
export async function updateOvertimeLog(id:string, userId:number, time:number, projectId:string, workLog:string ) {  
    let overTimeInfo = await selectOneDataByParam(TABLENAMEENUM.加班, {id}, []);

    if (!overTimeInfo || !overTimeInfo.id) throw new BizError(ERRORENUM.目标数据已不存在)
    if (overTimeInfo.state == STATED.是) throw new BizError(ERRORENUM.已经通过审批的加班不能修改或删除);

    await updateManyData(TABLENAMEENUM.加班, {id}, {projectId, time, workLog});

    return {isSuccess:true, message:"修改成功"};
}


/**
 * 删除加班
 * @param id 
 * @returns 
 */
export async function deleteOvertimeLog(id:string ) {
    let overTimeInfo = await selectOneDataByParam(TABLENAMEENUM.加班, {id}, []);

    if (!overTimeInfo || !overTimeInfo.id) throw new BizError(ERRORENUM.目标数据已不存在)
    if (overTimeInfo.state == STATED.是) throw new BizError(ERRORENUM.已经通过审批的加班不能修改或删除);

    await delData(TABLENAMEENUM.加班, {id});

    return {isSuccess:true, message:"删除成功"};
}



/**
 * 加班审批
 * @param id 
 * @param type 
 * @param notPassingStr 
 * @returns 
 */
export async function approveOverTime(id:string, type:number, notPassingStr:string) {
    if (type == APPROVETYPE.不通过 && !notPassingStr) {
        throw new BizError(ERRORENUM.参数错误);
    }
    if (type == APPROVETYPE.通过) notPassingStr = "";

    let overTimeInfo = await selectOneDataByParam(TABLENAMEENUM.加班, {id}, []);
    if (!overTimeInfo || !overTimeInfo.id) throw new BizError(ERRORENUM.目标数据已不存在);

    await updateManyData(TABLENAMEENUM.加班, {id}, {state:type, notPassingStr});

    return {isSuccess:true, message:"操作成功"};
}


/**
 * 审批列表
 * @param userId 
 * @returns 
 */
export async function approveOverTimeList(userId:number) {
    let includeConf = {};
    includeConf[TABLENAMEENUM.项目表] = ["name"];
    includeConf[TABLENAMEENUM.用户表] = ["name"];
    let column = ["projectId", "id", "time", "dateKey", "workLog"];
    let overTimeList = await selectDataToTableAssociation(TABLENAMEENUM.加班, includeConf, {approver:userId, state:STATED.否}, column);
    let dataList = [];
    let peopleMap = {};
    overTimeList.forEach(info => {
        let {user, id, time, dateKey, workLog, project} = info;
        let userName = user.name;
        dataList.push({
            dateKey:moment(dateKey).format("YYYY-MM-DD"),
            userName, projectName:project.name, id, time, workLog
        });
        peopleMap[userName] = 1;
    });
    
    return {dataList, peopleCount:Object.keys(peopleMap).length };
}