import { APPROVETYPE, ASKFORLEAVETYPE, DEPARTMENTENUM, SYSTEMSETKEY, TABLENAMEENUM } from "../config/enum";
import { selectDataListByParam, selectDataToTableAssociation, selectOneDataByParam } from "../model/findData";
import moment = require("moment");
import { updateManyData } from "../model/updateData";
import { getDBTime } from "../util/myTools";
import { BizError } from "../util/bizError";
import { ERRORENUM } from "../config/errorEnum";

/**
 * 成员列表
 * @returns
 */
export async function getMemberList() {
    let includeConf = {};
    includeConf[TABLENAMEENUM.部门表] = {colum:["name"], where:{departmentId:{"%in%":[DEPARTMENTENUM.开发, DEPARTMENTENUM.设计, DEPARTMENTENUM.项目经理, DEPARTMENTENUM.人事]}} };
    includeConf[TABLENAMEENUM.角色表] = {colum:["roleName"], where:{} };
    let column = ["userId", "name", "ed", "roleId", "departmentId", "annualLeave"];
    let userList = await selectDataToTableAssociation(TABLENAMEENUM.用户表, includeConf, {state:1}, column);

    let dataList = [];
    for (let i = 0; i < userList.length; i++) {
        let info = userList[i];
        let {userId, name, ed, department, annualLeave} = info;

        let monthDifference = 0;
        let annualLeaveCount = 0;
        let monthNum = new Date(ed).getMonth()
        let longStr = ``;

        if (ed) monthDifference = -1 * moment(ed).diff(moment(), 'months');
        else {
            ed = "-";
            longStr = "-";
        }

        if (monthDifference >= 12) {
            /**满一年 */
            let year = Math.floor(monthDifference/12);
            longStr = `${year}年`;
            if (monthDifference%12) longStr += `${monthDifference%12}个月`;
            annualLeaveCount = year + 5 > 10 ? 10 : year + 5;

        } else {
            /**未满一年 */
            longStr = `${monthDifference}个月`;
            annualLeaveCount = Math.round((12 - monthNum)/12*5);
        }
       
        let {cto, annual} = await findUserAskCount(userId);
        dataList.push({
            userId,
            name,
            departmentName:department.name,
            ed,
            longStr,
            annualLeaveCount,
            nowALeave:annual/8,
            cto
        });
    }

    return {dataList};
}

/**
 * 校验年假
 */
export async function eccMemberTime() {
    /**获取年假更新时间 */
    let askResetInfo = await selectOneDataByParam(TABLENAMEENUM.系统设置表, {id:SYSTEMSETKEY.年假重置时间}, ["value"]);
    if (!askResetInfo || !askResetInfo.value) throw new BizError(ERRORENUM.系统错误);

    let includeConf = {};
    includeConf[TABLENAMEENUM.部门表] = {colum:["name"], where:{departmentId:{"%in%":[DEPARTMENTENUM.开发, DEPARTMENTENUM.设计, DEPARTMENTENUM.项目经理, DEPARTMENTENUM.人事]}} };
    includeConf[TABLENAMEENUM.角色表] = {colum:["roleName"], where:{} };
    let column = ["userId", "name", "ed", "roleId", "departmentId", "annualLeave"];
    let userList = await selectDataToTableAssociation(TABLENAMEENUM.用户表, includeConf, {state:1}, column);
    let annMap = {};
    for (let i = 0; i < userList.length; i++) {
        let info = userList[i];
        let {userId, name, ed, department} = info;

        let monthDifference = 0;
        let annualLeaveCount = 0;
        let monthNum = new Date(ed).getMonth()
        let longStr = ``;

        if (ed) monthDifference = -1 * moment(ed).diff(moment(), 'months');
        else {
            ed = "-";
            longStr = "-";
        }

        if (monthDifference >= 12) {
            /**满一年 */
            let year = Math.floor(monthDifference/12);
            longStr = `${year}年`;
            if (monthDifference%12) longStr += `${monthDifference%12}个月`;
            annualLeaveCount = year + 5 > 10 ? 10 : year + 5;

        } else {
            /**未满一年 */
            longStr = `${monthDifference}个月`;
            annualLeaveCount = Math.round((12 - monthNum)/12*5);
        }
        annMap[userId] = annualLeaveCount * 8;
    }

    let askList = await selectDataListByParam(TABLENAMEENUM.请假, {state:APPROVETYPE.通过, type:2, startTime:{"%gt%":askResetInfo.value}}, [ "day", "userId", "time", "type"]);
    let nianjia = {};
    askList.forEach(info => {
        let {userId, time, day} = info;
        let itemCount = time + (day*8);
        if (!nianjia[userId]) nianjia[userId] = 0;
        nianjia[userId] += itemCount;
    });

    for (let key in annMap) {
        let dNum = annMap[key] - (nianjia[key] || 0);
        await updateManyData(TABLENAMEENUM.用户表, {userId:key}, {annualLeave:dNum});
    }

    console.log("数据校验成功");
}

/**
 * 更新入职时间
 * @param userId 
 * @param edTime 
 * @returns 
 */
export async function updateMemberData(userId:number, edTime:string) {
    let ed = getDBTime(edTime);
    await updateManyData(TABLENAMEENUM.用户表, {userId}, {ed});

    return {isSuccess:true};
}


/**
 * 用户下拉框
 * @returns 
 */
export async function memberSelectList() {
    let result = await selectDataListByParam(TABLENAMEENUM.用户表, {}, ["userId", "name"]);

    let dataList = [];
    result.forEach(info => {
        dataList.push({
            value:info.userId,
            label:info.name
        });
    });
    return {dataList}; 
}

/**
 * 部门用户下拉框
 * @param departmentId 
 * @returns 
 */
export async function getDepartmentUserList(departmentId:number) {
    let userList = await selectDataListByParam(TABLENAMEENUM.用户表, {departmentId}, ["userId", "name"]);
    let dataList = [];
    userList.forEach(info => {
        dataList.push({
            value:info.userId,
            label:info.name
        });
    });
    return {dataList};
}

/**
 * 获取用户剩余调休和剩余年假时长
 * @param userId 目标用户
 * @returns {result:{cto:调休, annual:年假}}
 */
export async function findUserAskCount(userId:number) {
    let askList = await selectDataListByParam(TABLENAMEENUM.请假, {userId, state:APPROVETYPE.通过}, [ "day", "time", "type"]);
    let tiaoXiu = 0;
    askList.forEach(info => {
        let {type, day, time} = info;
        let itemLong = 0;
        if (day != 0) itemLong += day*8;
        if (time != 0) itemLong += time;

        if (type == ASKFORLEAVETYPE.调休) {
            tiaoXiu += itemLong;
        }
    });

    let otList = await selectDataListByParam(TABLENAMEENUM.加班, {userId, state:APPROVETYPE.通过}, ["time"]);
    let ovCount = 0;//加班时间
    otList.forEach(info => {
        ovCount += info.time || 0;
    });

    let cto = ovCount - tiaoXiu;
    
    let thisUserInfo = await selectOneDataByParam(TABLENAMEENUM.用户表, {userId}, ["annualLeave"]);
    let annual = thisUserInfo.annualLeave || 0;
    return {cto, annual};
}


/**
 * 获取当前年假更新时间
 * @returns 
 */
export async function getASKMs() {
    let askSetInfo =  await selectOneDataByParam(TABLENAMEENUM.系统设置表, {id:SYSTEMSETKEY.年假重置时间}, ["value"]);
    if (!askSetInfo) throw new BizError(ERRORENUM.系统错误, '未初始化年假计算时间');
    let result = {
        ms:askSetInfo.value
    };
    return result;
}


/**
 * 更新年假时间
 * @param ms 
 * @returns 
 */
export async function resettingASKMs(ms:string) {
    await updateManyData(TABLENAMEENUM.系统设置表, {id:SYSTEMSETKEY.年假重置时间}, {value:ms});
    await eccMemberTime();
    return {isSuccess:true}
}