/**
 * 工时
 */
import moment = require("moment");
import { selectDataCountByParam, selectDataCountToTableAssociation, selectDataListByParam, selectDataToTableAssociation, selectDataToTableAssociationToPage, selectOneDataByParam } from "../model/findData";
import { DEPARTMENTENUM, TABLENAMEENUM } from "../config/enum";
import { ERRORENUM } from "../config/errorEnum";
import { BizError } from "../util/bizError";
import { getWorkingLogId } from "../util/myTools";
import { addData } from "../model/addData";
import { updateManyData } from "../model/updateData";
import { delData } from "../model/delData";
import { changeEnumValue } from "../util/piecemeal";



/**
 * 填报工时
 * @param userId 填报人
 * @param time 工时
 * @param projectId 项目号
 * @param isOutside 是否外出 true/false
 * @param date 填报时间
 * @param workLog 工作内容
 * @returns {isSuccess:true, message:"添加成功"}
 */
export async function addWorking(userId:number, time:number, projectId:string, outside:boolean, date:string, workLog:string ) {
    let dateKey = moment(date).format("YYYY-MM-DD")+ ' 10:00:00';
    let selectStartTime =  moment(date).format("YYYY-MM-DD")+ ' 00:00:00';
    let selectEndTime =  moment(date).format("YYYY-MM-DD")+ ' 23:00:00';
    let thisDayLogList = await selectDataListByParam(TABLENAMEENUM.工时表, {userId, dateKey:{"%between%":[selectStartTime, selectEndTime]}}, ["time"]);
    let thisDayLogCount = 0;
    thisDayLogList.forEach(info => {
        thisDayLogCount += info.time;
    });
    if ((thisDayLogCount + time) > 8) throw new BizError(ERRORENUM.一天只有8小时多余的时间请填加班, `${thisDayLogCount + time } >= 24`);
    // if (logCount>0) throw new BizError(ERRORENUM.重复提交, `${DateKey}工时填报重复提交`);

    //查找项目编号
    let projectInfo = await selectOneDataByParam(TABLENAMEENUM.项目表, {}, ["projectId", "name"]);
    if ( !projectInfo.projectId ) throw new BizError(ERRORENUM.不存在该项目, `${userId} 填报 ${projectId}`);

    let id = getWorkingLogId(userId);

    let isOutside = outside ? 1 : 0;

    let addInfo = {id, projectId, userId, time, dateKey, isOutside, workLog};
    await addData(TABLENAMEENUM.工时表, addInfo);

    return {isSuccess:true, message:"添加成功"};
}


/**
 * 查看本月填报的工时  工时列表
 * @param userId 填报人
 * @returns { dataList:本周填报的工时信息, 本周工时统计 }
 */
export async function workinglogList(userId:number) {
    let s = moment().startOf('month').format('YYYY-MM-DD') + ' 00:00:00';// 周-日期
    let e = moment().format('YYYY-MM-DD') + ' 23:59:59';// 周日期

    let selectParam = {
        userId,
        dateKey:{"%between%":[s, e]}
    };
    let includeConf = {};
    includeConf[TABLENAMEENUM.项目表] = ["name"];
    let column = ["dateKey", "time", "isOutside", "workLog", "projectId", "id"];
    let thisWeekLog = await selectDataToTableAssociation(TABLENAMEENUM.工时表, includeConf, selectParam, column);

    thisWeekLog.sort((a, b) => {return new Date(b.dateKey).valueOf() - new Date(a.dateKey).valueOf() });

    let timeCount = 0;
    let distinctMap = {};
    let dataList = [];
    thisWeekLog.forEach(info => {
        let {dateKey, project, isOutside, time, workLog, projectId, id} = info;
        let dateKeyStr = moment(dateKey).format("YYYY-MM-DD");
        dataList.push({
            dateKey:dateKeyStr,
            projectName:project.name,
            isOutside:isOutside == 1,
            time,
            workLog,
            projectId,
            id
        });
        timeCount += time;
        distinctMap[dateKeyStr] = 1;
    });
    
    return { dataList:dataList, timeCount, dateCount:Object.keys(distinctMap).length };
}


/**
 * 修改已填报的工时信息
 * @param userId 填报人
 * @param id 属性的数据类型
 * @param time 工时
 * @param projectId 项目号
 * @param isOutside 是否外出 true/false
 * @param workLog 工作内容
 * @returns { isSuccess:true, message:"修改成功" }
 */
export async function updateWorkinglog(userId:number, id:string, time:number, projectId:string, outside:boolean, workLog:string) {
    let itemInfo = await selectOneDataByParam(TABLENAMEENUM.工时表, {id}, []);
    if (!itemInfo.id) throw new BizError(ERRORENUM.目标数据已不存在, id);

    let dbList = await selectDataListByParam(TABLENAMEENUM.工时表, {userId, dateKey:itemInfo.dateKey}, []);

    let count = time;
    dbList.forEach(info => {
        if (info.id != id) {
            count += info.time;
        }
    });
    if (count > 8) throw new BizError(ERRORENUM.一天只有8小时多余的时间请填加班, `${ count } > 8`);

    let isOutside = outside ? 1 : 0;
    await updateManyData(TABLENAMEENUM.工时表, {id}, {projectId, time, workLog, isOutside} );

    return { isSuccess:true, message:"修改成功" };
}


/**
 * 根据传入id删除填报工时
 * @param userId 填报人
 * @param id 属性的数据类型
 * @returns { isSuccess:true, message:"删除成功" }
 */
export async function delWorkinglog(userId:number, id:string,) {
    let itemInfo = await selectOneDataByParam(TABLENAMEENUM.工时表, {id}, []);

    if (!itemInfo.id) throw new BizError(ERRORENUM.目标数据已不存在, id);

    await delData(TABLENAMEENUM.工时表, {id});

    return { isSuccess:true, message:"删除成功" };
}


/**
 * 工时管理列表 按时间分
 * @param page 页数
 * @param startMs 开始时间
 * @param endMs 结束时间
 * @returns {dataList:工时管理列表, total:工时列表统计 }
 */
export async function adminWorkLogForTime(page:number, startMs:string, endMs:string) {
    let selectParam:any = {};
    if (startMs && endMs) {
        selectParam.dateKey = {"%between%":[moment(startMs).format("YYYY-MM-DD") + ' 00:00:00', moment(endMs).format("YYYY-MM-DD") + ' 23:59:59']};
    }

    let includeConf = {};
    includeConf[TABLENAMEENUM.项目表] = ["name"];
    includeConf[TABLENAMEENUM.用户表] = ["name", "departmentId" ];
    
    let column = ["dateKey", "time", "isOutside"];
    let dbList = await selectDataToTableAssociationToPage(TABLENAMEENUM.工时表, includeConf, selectParam, column, page, 10  );

    let dataList = [];

    let distinctMap = {};
    dbList.forEach(info => {
        let {dateKey, users, project } = info;
        let year = moment(dateKey).year();
        let month = moment(dateKey).month()+1;
        let day = moment(dateKey).date();
        dataList.push({
            year,
            month,
            day,
            name:users.name,
            departmentName : changeEnumValue(DEPARTMENTENUM, users.departmentId),
            projectName:project.name,
            projectId:project.projectId,
            workLog:info.workLog,
            time: info.time
        });
        distinctMap[day] = 1;
    });

    return {dataList, total:dbList.length, dateCount:Object.keys(distinctMap).length };
}


/**
 * 工时管理列表 按项目分
 * @param page 页数
 * @param startMs 开始时间
 * @param endMs 结束时间
 * @returns {dataList:工时管理列表, total:工时列表统计 };
 */
export async function adminWorkLogForProject(page:number, startMs:string, endMs:string) {
    let selectParam:any = {};
    if (startMs && endMs) {
        selectParam.dateKey = {"%between%":[moment(startMs).format("YYYY-MM-DD") + ' 00:00:00', moment(endMs).format("YYYY-MM-DD") + ' 23:59:59']};
    }
    
    let includeConf = {};
    includeConf[TABLENAMEENUM.项目表] = ["name", "projectId"];
    
    let column = ["time", "isOutside", "projectId"];
    let dbList = await selectDataToTableAssociation(TABLENAMEENUM.工时表, includeConf, selectParam, column );

    let distinctMap = {};// => {name:{id:"", count:0}}

    dbList.forEach(info => {
        let {project, time} = info;
        if (!distinctMap[project.name]) distinctMap[project.name] = {id:project.projectId, count:0 }
        distinctMap[project.name].count += time;
    });

    let result = [];
    for (let name in distinctMap) {
        let {id, count} = distinctMap[name];
        let dayCount = Math.ceil(count/8 * 100)/100
        result.push({
            name,
            id,
            hostCount:count,
            dayCount
        });
    }

    return {dataList:result.splice((page-1)*10, 10 ), total:result.length };
}


/**
 * 导出按时间分的工时管理列表
 * @param startMs 开始时间
 * @param endMs 结束时间
 * @returns {dataList:工时管理列表, name:`【按时间分工时】.xlsx`}
 */
export async function outPutAdminWorkingLogListForTime(startMs:string, endMs:string) {
    let selectParam:any = {};
    if (startMs && endMs) {
        selectParam.dateKey = {"%between%":[moment(startMs).format("YYYY-MM-DD") + ' 00:00:00', moment(endMs).format("YYYY-MM-DD") + ' 23:59:59']};
    }

    /**加班 */
    let overtimeList = await selectDataListByParam(TABLENAMEENUM.加班, selectParam, ["dateKey", "userId", "time" ]);
    let overtimeMap = {};
    overtimeList.forEach(info => {
        let {dateKey, userId, time} = info;
        let dateKeyStr = moment(dateKey).format("YYYY-MM-DD");
        if (!overtimeMap[dateKeyStr]) overtimeMap[dateKeyStr] = {};
        if (!overtimeMap[dateKeyStr][userId]) overtimeMap[dateKeyStr][userId] = time;
    });

    /**工时 */
    let includeConf = {};
    includeConf[TABLENAMEENUM.项目表] = ["name"];
    includeConf[TABLENAMEENUM.用户表] = ["name", "userId",  "departmentId" ];
    let column = ["dateKey", "time", "isOutside"];
    let dbList = await selectDataToTableAssociation(TABLENAMEENUM.工时表, includeConf, selectParam, column );

    let result = [
        ["日期年","日期月","日期日","部门","姓名","项目名称","项目编号","时长/小时","工作内容", "加班时长"]
    ];

    dbList.forEach(info => {
        let {dateKey, users, project } = info;
        let year = moment(dateKey).year();
        let month = moment(dateKey).month()+1;
        let day = moment(dateKey).date();
        let itemList = [
            year,
            month,
            day,
            changeEnumValue(DEPARTMENTENUM, users.departmentId),
            users.name,
            project.name,
            project.projectId,
            info.time,
            info.workLog
        ];
        if (overtimeMap[dateKey][users.userId]) itemList.push(overtimeMap[dateKey][users.userId]);
        result.push(itemList);
    });

    return {dataList:result, name:`【按时间分工时】.xlsx`};
}


/**
 * 导出按项目分的工时管理列表
 * @param startMs 开始时间
 * @param endMs 结束时间
 * @returns {dataList:工时管理列表, name:`【按项目分工时】.xlsx`}
 */
export async function outPutAdminWorkingLogListForProject(startMs:string, endMs:string) {
    let selectParam:any = {};
    if (startMs && endMs) {
        selectParam.dateKey = {"%between%":[moment(startMs).format("YYYY-MM-DD") + ' 00:00:00', moment(endMs).format("YYYY-MM-DD") + ' 23:59:59']};
    }

    /**加班 */
    let overtimeList = await selectDataListByParam(TABLENAMEENUM.加班, selectParam, [ "projectId", "time" ]);
    let overtimeMap = {};
    overtimeList.forEach(info => {
        let {projectId, time} = info;
        if (!overtimeMap[projectId]) overtimeMap[projectId] = 0;
        overtimeMap[projectId] += time;
    });

    let includeConf = {};
    includeConf[TABLENAMEENUM.项目表] = ["name", "projectId"];
    let column = ["time"];
    let dbList = await selectDataToTableAssociation(TABLENAMEENUM.工时表, includeConf, selectParam, column );
    let distinctMap = {};// => {name:{id:"", count:0}}
    dbList.forEach(info => {
        let {project, time} = info;
        if (!distinctMap[project.name]) distinctMap[project.name] = {id:project.projectId, count:0 };
        distinctMap[project.name].count += time;
    });

    let result = [
        ["项目名称","项目编号","时长/小时","时长/人天", "加班时长/小时"]
    ];
    for (let name in distinctMap) {
        let {id, count} = distinctMap[name];
        let dayCount = Math.ceil(count/8 * 100)/100;
        let itemList = [
            name,
            id,
            count,
            dayCount
        ];
        if (overtimeMap[id]) itemList.push(overtimeMap[id]);

        result.push(itemList);
    }

    return {dataList:result, name:`【按项目分工时】.xlsx`};
}


/**
 * 部门主管看的工时列表
 */
export async function departmentWorklog(departmentId:number, startTime:string, endTime:string, userId:number, page:number, projectId:string) {

    let selectParam:any = {};
    if (startTime && endTime) {
        selectParam.dateKey = {"%between%":[moment(startTime).format("YYYY-MM-DD") + ' 00:00:00', moment(endTime).format("YYYY-MM-DD") + ' 23:59:59']};
    }
    if (userId) selectParam.userId = userId;
    if (projectId) selectParam.projectId = projectId;

    let includeConf = {};
    includeConf[TABLENAMEENUM.项目表] = ["name"];
    includeConf[TABLENAMEENUM.用户表] = { column:["name", "departmentId" ], where:{departmentId} }
    let column = ["dateKey", "time", "isOutside", "workLog"];
    let dbList = await selectDataToTableAssociationToPage(TABLENAMEENUM.工时表, includeConf, selectParam, column, page, 10 );

    let count = await selectDataCountToTableAssociation(TABLENAMEENUM.工时表, includeConf, selectParam);
    let dataList = [];
    dbList.forEach(item => {
        let {user, project, dateKey, time, isOutside, workLog} = item;
        dataList.push({
            dateKey:moment(dateKey).format("YYYY-MM-DD"),
            name:user.name,
            time:time,
            projectName:project.name,
            isOutside:isOutside,
            workLog:workLog
        });
    }); 

    return {dataList, count}
}
