/**
 * 请假逻辑
 */
import moment = require("moment");
import * as myTools from "../util/myTools";
import { APPROVETYPE, ASKFORLEAVETYPE, ROLEENUM, STATED, SYSTEMCONSTCONFIGENUM, TABLENAMEENUM } from "../config/enum";
import { queue } from "./system";
import { BizError } from "../util/bizError";
import { ERRORENUM } from "../config/errorEnum";
import { selectDataCountByParam, selectDataListByParam, selectDataToTableAssociation, selectDataToTableAssociationToPage, selectOneDataByParam, selectOneDataToTableAssociation } from "../model/findData";
import { addData } from "../model/addData";
import { changeEnumValue } from "../util/piecemeal";
import { delData } from "../model/delData";
import { updateManyData } from "../model/updateData";
import { findUserAskCount } from "./member";


/**
 * 添加请假记录
 * @param userId 填报人
 * @param userInfo 
 * @param des 描述
 * @param startTime 开始请假时间
 * @param endTime 结束请假时间
 * @param type 请假类型
 * @param day 填报时间
 * @param time 
 * @returns 
 */
export async function addAskForLeaveLog(userId:number, userInfo, des:string, startTime:string, endTime:string, type:number, day:number, time:number) {
    /**审批人 */
    let approver = 0; 
    if (userInfo.roleId == ROLEENUM.员工) {
        approver = queue[SYSTEMCONSTCONFIGENUM.部门经理][userInfo.departmentId].userId;
    } else if (userInfo.roleId == ROLEENUM.部门经理) {
        approver = queue[SYSTEMCONSTCONFIGENUM.请假默认].userId;
    }
    
    /**年假必须是8的整数倍 */
    if (type == ASKFORLEAVETYPE.年假) {
        let timeCount = (day*8) + time;
        if (timeCount%4 != 0) throw new BizError(ERRORENUM.年假限制为4小时的倍数)
        if (timeCount > userInfo.annualLeave) throw new BizError(ERRORENUM.年假剩余不足请重新分配请假时间);
    } else if (type == ASKFORLEAVETYPE.调休) {
        let askList = await selectDataListByParam(TABLENAMEENUM.请假, {userId, type:ASKFORLEAVETYPE.调休}, ["day", "time"] );
        let askCount = 0;
        askList.forEach(item => {
            let {day, time} = item;
            if (day!=0) askCount += day*8;
            if (time != 0) askCount += time;
        });
        let ovList = await selectDataListByParam(TABLENAMEENUM.加班, {userId, state:APPROVETYPE.通过}, ["time"] );
        let ovCount = 0;
        ovList.forEach(item => {
            let {time} = item;
            ovCount += time;
        });
        let residue = ovCount - askCount;
        let timeCount = (day*8) + time;
        if (timeCount > residue) throw new BizError(ERRORENUM.调休剩余不足请重新分配请假时间)
    }

    let addInfo = {
        id : myTools.generateAskForLeaveId(userId),
        startTime:myTools.getDBTime(startTime),
        endTime:myTools.getDBTime(endTime),
        des,
        userId,
        approval:approver,
        time,
        day,
        type,
        dateKey:myTools.generateDateKey(),
        state:0
    };
    await addData(TABLENAMEENUM.请假, addInfo);

    return {isSuccess:true, message:"添加成功"};
}


/**
 * 我的请假列表
 * @param userId 用户id 
 */
export async function myAskForLeaveList(userId:number) {

    let column = ["id", "des", "time", "type", "day", "state", "notPassing", "dateKey", "startTime", "endTime"];
    let askList = await selectDataListByParam(TABLENAMEENUM.请假, {userId}, column);

    let notApproveList = [];
    let notPassingList = [];
    let passingList = [];

    askList.forEach(info => {
        let { id, des, time, type, day, state, notPassing, dateKey } = info;
        let changeType = changeEnumValue(ASKFORLEAVETYPE, type);

        let startTime = myTools.getClientTime(info.startTime);
        let endTime = myTools.getClientTime(info.endTime);
        let timeStr = `${day}日${time}小时`;

         if (new Date(dateKey).valueOf() > ( new Date().valueOf() - (3600*1000*24*80)) ) {
            //未超过80天
            if (state == STATED.否) notApproveList.push({id, des, startTime, endTime, timeStr, changeType, type, day, time});
            else if (state == APPROVETYPE.通过) passingList.push({des, startTime, endTime, timeStr, changeType});
            else if (state == APPROVETYPE.不通过) notPassingList.push({des, startTime, endTime, timeStr, changeType, notPassing});
        } else {
            //超过80天
            if (state == STATED.否) notApproveList.push({id, des, startTime, endTime, timeStr, changeType, type, day, time});
        }
    });

    let {cto, annual} = await findUserAskCount(userId);

    return {
        notApproveList, 
        notPassingList, 
        passingList, 
        residueDay:Math.floor(cto/8),
        residueTime: cto%8,
        annualLeaveDay :Math.floor( annual/8),
        annualLeaveTime : annual%8,
    };
}




/**
 * 删除未通过审核的请假记录
 * @param id 
 */
export async function deleteAskForLeave(id:string) {
    let askForLeaveInfo = await selectOneDataByParam(TABLENAMEENUM.请假, {id}, ["state"]);

    if (askForLeaveInfo.state == STATED.是) throw new BizError(ERRORENUM.已经通过审批的请假不能修改或删除);

    await delData(TABLENAMEENUM.请假, {id});

    return {isSuccess:true, message:"删除成功"}; 
}


/**
 * 修改 未审批的请假记录
 * @param id 
 * @param userId 
 * @param des 
 * @param startTime 
 * @param endTime 
 * @param type 
 * @param day 
 * @param time 
 * @returns 
 */
export async function updateAskForLeaveLog(id:string, userId:number, des:string, startTime:string, endTime:string, type:number, day:number, time:number) {
    let askForLeaveInfo = await selectOneDataByParam(TABLENAMEENUM.请假, {id}, []);
    if (!askForLeaveInfo || !Object.keys(askForLeaveInfo).length) {
        throw new BizError(ERRORENUM.目标数据已不存在);
    }
    if (askForLeaveInfo.state == STATED.是) throw new BizError(ERRORENUM.已经通过审批的数据不能修改);
    let updateInfo = {des,  startTime:myTools.getDBTime(startTime), endTime:myTools.getDBTime(endTime), type, day, time};
    await updateManyData(TABLENAMEENUM.请假, {id}, updateInfo);

    return {isSuccess:true, message:"修改"};
}


/**
 * 审批 请假
 * @param id 
 * @param type 
 * @param notPassingStr 
 * @returns 
 */
export async function approveAskForLeave(id:string, type:number, notPassingStr:string) {
    let askForLeaveInfo = await selectOneDataByParam(TABLENAMEENUM.请假, {id}, []);
    if (!askForLeaveInfo || !Object.keys(askForLeaveInfo).length) {
        throw new BizError(ERRORENUM.目标数据已不存在);
    }
    if (askForLeaveInfo.state == STATED.是) {
        throw new BizError(ERRORENUM.该请假已通过审批);
    }

    if (type == APPROVETYPE.通过) {
        notPassingStr = "";

        let { cto, annual} = await findUserAskCount(askForLeaveInfo.userId);
        let askLong = ((askForLeaveInfo.day * 8 ) + askForLeaveInfo.time);
        if (askForLeaveInfo.type == ASKFORLEAVETYPE.年假) { 
            //二次校验时间是否满足请假
            let annualLeave = annual - askLong;
            if (annualLeave < 0) throw new BizError(ERRORENUM.年假剩余时间不足);
    
            await updateManyData(TABLENAMEENUM.用户表, {annualLeave:annualLeave}, {userId:askForLeaveInfo.userId});
        } else {
            /**调休 */
            let residuedCTO = cto - askLong;
            if (residuedCTO < 0) throw new BizError(ERRORENUM.调休时间不足);
        }

    } else {/**不通过 */
        if (!notPassingStr) throw new BizError(ERRORENUM.参数错误);
    }

    await updateManyData(TABLENAMEENUM.请假, {id}, {state:type, notPassing:notPassingStr});

    return {isSuccess:true, message:"操作成功"};
}


/**
 * 审批列表
 * @param userId 用户id
 * @returns {result:{dataList:"数据列表", peopleCount:"请假人数"}}
 */
export async function approveAskForLeaveList(userId:number) {
    let includeConf = {};
    includeConf[TABLENAMEENUM.用户表] = ["name"];
    let column = ["startTime", "endTime", "day", "time", "type", "des", "id"];
    let askForLeaveList = await selectDataToTableAssociation(TABLENAMEENUM.请假, includeConf, {approval:userId, state:STATED.否}, column);
    let dataList = [];
    let peopleMap = {};
    askForLeaveList.forEach(info => {
        let {user, startTime, endTime, day, time, type, des, id} = info;
        let userName = user.name;
        dataList.push({
            startTime:myTools.getClientTime(startTime),
            endTime:myTools.getClientTime(endTime),
            timeStr : `${day}日${time}小时`,
            type:changeEnumValue(ASKFORLEAVETYPE, type), 
            des,
            userName,
            id
        });
        peopleMap[userName] = 1;
    });

    return {dataList, peopleCount:Object.keys(peopleMap).length };
}


/**
 * 人事的请假列表总览
 * @param page 
 * @param userId 
 * @param startTime 
 * @param endTime 
 */
export async function getPersonnelAFLList(page:number, userId:number, startTime:string, endTime:string) {
    let selectParam:any = {};
    if (userId) selectParam.userId = userId;
    if (startTime) selectParam.startTime = {"%gt%":moment( startTime).format("YYYY-MM-DD")+" 00:00:00"};
    if (endTime) selectParam.endTime =  {"%lt%":moment( endTime).format("YYYY-MM-DD")+" 00:00:00"};

    let includeConf = {};
    includeConf[TABLENAMEENUM.用户表] = ["name", "roleId", "departmentId"];
    let column = ["startTime", "endTime", "day", "time", "type", "des", "id", "state"];
    let dbList = await selectDataToTableAssociationToPage(TABLENAMEENUM.请假, includeConf, selectParam, column, page, 10 );
    let count = await selectDataCountByParam(TABLENAMEENUM.请假, selectParam);
    let dataList = [];
    
    dbList.forEach(info => {
        let { startTime, endTime, day, time, type, des, state, user } = info;

        let approver = "";
        if (user.roleId == ROLEENUM.员工) {
            approver = queue[SYSTEMCONSTCONFIGENUM.部门经理][user.departmentId].name;
        } else if (user.roleId == ROLEENUM.部门经理) {
            approver = queue[SYSTEMCONSTCONFIGENUM.请假默认].name;
        }
    
        let stateStr = "";
        if (state == 0) stateStr = "未审批";
        else if (state == 1) stateStr = "已通过";
        else stateStr = "未通过";
        dataList.push({
            name:user.name, 
            startTime:myTools.getClientTime(startTime), 
            endTime:myTools.getClientTime(endTime), 
            long:(day*8)+ time,
            type:changeEnumValue(ASKFORLEAVETYPE, type), des, 
            state:stateStr, 
            approval:approver
        });
    });

    return {dataList, count};
}


/**
 * 人事导出请假列表
 * @param userId 
 * @param startTime 
 * @param endTime 
 */
export async function outPutPersonnelAFLList(userId:number, startTime:string, endTime:string) {
    let selectParam:any = {};
    if (userId) selectParam.userId = userId;
    if (startTime) selectParam.startTime = {"%gt%":moment( startTime).format("YYYY-MM-DD")+" 00:00:00"};
    if (endTime) selectParam.endTime =  {"%lt%":moment( endTime).format("YYYY-MM-DD")+" 00:00:00"};

    let includeConf = {};
    includeConf[TABLENAMEENUM.用户表] = ["name", "roleId", "departmentId"];
    let column = ["startTime", "endTime", "day", "time", "type", "des", "id", "state"];

    let dbList = await selectDataListByParam(TABLENAMEENUM.请假, selectParam, column);
    let dataList = [];
    dbList.forEach(info => {
        let {users, startTime, endTime, day, time, type, des, state } = info;

        let approver = "";
        if (users.roleId == ROLEENUM.员工) {
            approver = queue[SYSTEMCONSTCONFIGENUM.部门经理][users.departmentId].name;
        } else if (users.roleId == ROLEENUM.部门经理) {
            approver = queue[SYSTEMCONSTCONFIGENUM.请假默认].name;
        }

        let stateStr = "";
        if (state == 0) stateStr = "未审批";
        else if (state == 1) stateStr = "已通过";
        else stateStr = "未通过";
        dataList.push({
            name:users.name, 
            startTime:myTools.getClientTime( startTime), 
            endTime:myTools.getClientTime( endTime), 
            long:(day*8)+ time,
            type:changeEnumValue(ASKFORLEAVETYPE, type), des, 
            state:stateStr, 
            approval:approver
        });
    });
    return {dataList};
}