/**
 * strategyFactory.ts
 * 该文件定义了策略工厂类，用于创建各种数据策略。
 */

import { dataStrategy } from "./dataStrategy";

// 导入具体策略类
import { sightVisitorFlowByDayStrategy } from "./map1/strategies/left/sightVisitorFlowByDayStrategy";
import { gateStatusStrategy } from "./map1/strategies/left/gateStatusStrategy";
import { sightVisitorFlowByHourStrategy } from "./map1/strategies/left/sightVisitorFlowPerHourStrategy";
import { guchengLoadStrategy } from "./map1/strategies/left/guchengLoadStrategy";
import { totalVisitorFlowStrategy } from "./map1/strategies/left/totalVisitorFlowStrategy";
import { totalVisitorFlowByHourStrategy } from "./map1/strategies/left/totalVisitorFlowByHourStrategy";

import { currentEventStrategy } from "./map1/strategies/middle/currentEventStrategy";
import { totalEventCountStrategy } from "./map1/strategies/middle/totalEventCountStrategy";

import { getEventCountByYearStrategy } from "./map1/strategies/right/getEventCountByYearStrategy";
import { eventTimeDistributionStrategy } from "./map1/strategies/right/eventTimeDistributionStrategy";
import { eventMonthDistributionStrategy } from "./map1/strategies/right/eventMonthDistributionStrategy";
import { eventCategoryCountStrategy } from "./map1/strategies/right/eventCategoryCountStrategy";
import { eventSubCategoryCountStrategy } from "./map1/strategies/right/eventSubCategoryCountStrategy";
import { eventSrcStrategy } from "./map1/strategies/right/eventSrcStrategy";
import { gridEventCountStrategy } from "./map1/strategies/right/gridEventCountStrategy";
import { eventProcessingTimeStrategy } from "./map1/strategies/right/eventProcessingTimeStrategy";
import {allEventDataStrategy} from "./map1/strategies/middle/eventDataStrategy";
import monthlyVisitorCountStrategy from "./map2/strategies/left/monthlyVisitorCountStrategy";
import sightVisitorRankStrategy from "./map2/strategies/left/sightVisitorRankStrategy";

/**
 * 策略工厂类，用于创建和管理各种数据策略。
 */
export class strategyFactory {
    /**
     * 存储策略类型与其对应类的映射。
     */
    private static strategies: { [key: string]: new () => dataStrategy } = {
        // map 1
        'allEvents': allEventDataStrategy,
        'sightVisitorFlowByDay': sightVisitorFlowByDayStrategy,
        'gateStatus': gateStatusStrategy,
        'sightVisitorFlowPerHour': sightVisitorFlowByHourStrategy,
        'guchengLoad': guchengLoadStrategy,
        'totalVisitorFlow': totalVisitorFlowStrategy,
        'totalVisitorFlowByHour': totalVisitorFlowByHourStrategy,
        'getCurrentEventCount': currentEventStrategy,
        'totalEventCount': totalEventCountStrategy,
        'getEventCountByYear': getEventCountByYearStrategy,
        'getEventTimeDistribution': eventTimeDistributionStrategy,
        'getEventMonthDistribution': eventMonthDistributionStrategy,
        'getEventCategoryCount': eventCategoryCountStrategy,
        'getEventSubCategoryCount': eventSubCategoryCountStrategy,
        'getEventSourceCount': eventSrcStrategy,
        'getGridEventCount': gridEventCountStrategy,
        'getEventProcessingTime': eventProcessingTimeStrategy,

        // map2
        'monthlyVisitor': monthlyVisitorCountStrategy,
        'sightVisitorRank': sightVisitorRankStrategy
    };

    /**
     * 创建并返回指定类型的策略实例。
     * @param type - 策略类型的字符串标识。
     * @returns 创建的策略实例。
     * @throws 如果策略类型无效则抛出错误。
     */
    static createStrategy(type: string): dataStrategy {
        const StrategyClass = this.strategies[type];
        if (!StrategyClass) {
            throw new Error('Invalid strategy type.');
        }
        return new StrategyClass();
    }

    /**
     * 注册一个新的策略。
     * @param type - 策略类型的字符串标识。
     * @param strategy - 策略类。
     */
    static registerStrategy(type: string, strategy: new () => dataStrategy) {
        this.strategies[type] = strategy;
    }
}
