/**
 * getEventCountByYearStrategy.ts
 * 该文件定义了按年统计事件数量策略的具体实现。
 */

import { abstractDataStrategyRight } from "./abstractDataStrategyRight";
import excelSerialToJSDate from "../../../../util/excelDateToJSDate";
import moment = require("moment");

/**
 * 按年统计事件数量策略类，继承自 abstractDataStrategyRight。
 */
export class getEventCountByYearStrategy extends abstractDataStrategyRight {

    /**
     * 执行按年统计事件数量策略。
     * @param params - 请求参数，包含年份信息。
     * @returns 事件数量统计的对象表示。
     */
    execute(params?: any): any {
        // 检查参数是否有效
        this.paramsCheck(params);

        // 获取指定年份的事件数量统计，并转换为对象
        return this.getEventCountForYear(this.eventData, params.query.year);
    }

    /**
     * 获取指定年份的事件数量统计。
     * @param data - 从数据源提取的数据。
     * @param year - 指定年份 ("thisyear" or "lastyear")。
     * @returns 事件数量统计的对象表示。
     */
    private getEventCountForYear(data: any, year: string) {
        const currentYear = new Date().getFullYear();
        const targetYear = year === 'thisyear' ? currentYear : currentYear - 1;

        // 定义事件列表
        const eventList = ['事件总数', '历史遗留', '办结率'];
        const eventCount = this.registerItems(eventList);

        let finishedEventCount = 0;
        data.forEach(row => {
            // const rawDate = row['创建时间'];

            const rowDate = excelSerialToJSDate(row['创建时间']);

            let rowYear = 0;
            if (!isNaN(rowDate.getTime())) {
                try {
                    rowYear = rowDate.getFullYear();
                } catch (e) {
                    rowYear = 0;
                }
            }

            if (rowYear == targetYear) {
                eventCount.set('事件总数', (eventCount.get('事件总数') || 0) + 1);
                if (row['处置状态'] == '已办结') {
                    finishedEventCount++;
                }
            } else if (rowYear == targetYear - 1 && row['处置状态'] != '已办结') {
                eventCount.set('历史遗留', (eventCount.get('历史遗留') || 0) + 1);
            }
        });

        let totalEvents = eventCount.get('事件总数');

        eventCount.set('办结率', totalEvents ? (finishedEventCount / totalEvents) * 100 : 0);
        
        return this.convertToObject(eventCount);
    }

    /**
     * 将 Map 对象转换为带单位的对象数组
     * @param map - 要转换的 Map 对象
     * @returns 带单位的对象数组
     */
    private convertToObject(map: Map<string, number>): any {
        const result: Array<{ key: string, value: number, unit: string }> = [];
        map.forEach((value, key) => {
            const unit = key === '办结率' ? '%' : '件';
            result.push({ key: key, value: value, unit: unit });
        });
        return result;
    }
}
