/**
 * totalVisitorFlowByHourStrategy.ts
 * 该文件定义了按小时计算的总游客流量策略的具体实现，获取指定日期整个景区的各类人群客流量分布
 */

import excelSerialToJSDate from "../../../util/excelDateToJSDate";
import mapToObj from "../../../util/mapToObj";
import {abstractDataStrategyLeft} from "./abstractDataStrategyLeft";

/**
 * 按小时计算的总游客流量策略类，继承自 abstractDataStrategyLeft。
 */
export class totalVisitorFlowByHourStrategy extends abstractDataStrategyLeft {

    /**
     * 执行按小时计算的总游客流量策略。
     * @param params - 请求参数，包含日期信息。
     * @returns 全景区各类每小时各类游客流量。
     */
    execute(params?: any): any {
        if (!params || !params.query || !params.query.date) {
            throw new Error('Date parameter is required.');
        }
        const sightData = this.extractor.getData(totalVisitorFlowByHourStrategy.FILENAME, totalVisitorFlowByHourStrategy.SHEETNAME);
        return mapToObj(this.getTotalVisitorFlowByHour(sightData, params.query.date));
    }

    /**
     * 获取指定日期的每小时总游客流量。
     * @param data - 从数据源提取的数据。
     * @param date - 指定日期。
     * @returns 全景区各类每小时各类游客流量。
     */
    private getTotalVisitorFlowByHour(data: any, date: string) {
        const visitorCount: Map<string, Map<number, number>> = new Map();
        visitorCount.set('儿童', new Map());
        visitorCount.set('其他', new Map());
        visitorCount.set('学生', new Map());
        visitorCount.set('老人', new Map());

        visitorCount.forEach((value, key) => {
            for (let hour = 0; hour < 24; hour++) {
                value.set(hour, 0);
            }
        });

        data.forEach(row => {
            const rowDate = excelSerialToJSDate(row['游玩时间']);
            let rowDateString;
            try {
                rowDateString = rowDate.toISOString().split('T')[0];
            }
            catch (e) {
                rowDateString = 'invalid time';
            }

            rowDate.setHours(rowDate.getHours() - 8);
            const rowHour = rowDate.getHours();

            if (rowDateString == date) {
                visitorCount.set(row['订单游客类型'],visitorCount.get(row['订单游客类型']).set(rowHour, (visitorCount.get(row['订单游客类型']).get(rowHour) || 0) + 1));
            }
        });
        const ret: Map<string, object> = new Map();
        visitorCount.forEach((value, key) => {
            ret.set(key, mapToObj(value, "hour", "count"));
        })
        return ret;
    }

}