/**
 * eventMonthDistributionStrategy.ts
 * 该文件定义了事件按月分布策略的具体实现。
 */

import { abstractDataStrategyRight } from "./abstractDataStrategyRight";
import mapToObj from "../../../util/mapToObj";
import excelSerialToJSDate from "../../../util/excelDateToJSDate";

/**
 * 事件按月分布策略类，继承自 abstractDataStrategyRight。
 */
export class eventMonthDistributionStrategy extends abstractDataStrategyRight {
    /**
     * 执行事件按月分布策略。
     * @param params - 请求参数，包含年份信息。
     * @returns 事件按月分布计数的对象表示。
     */
    execute(params?: any): any {
        // 检查参数是否有效
        this.paramsCheck(params);

        // 获取指定年份的事件按月分布计数，并转换为对象
        return mapToObj(this.getEventMonthDistribution(this.eventData, params.query.year), "month", "count");
    }

    /**
     * 获取指定年份的事件按月分布计数。
     * @param data - 从数据源提取的数据。
     * @param year - 指定年份 ("thisyear" or "lastyear")。
     * @returns 事件按月分布计数的映射。
     */
    private getEventMonthDistribution(data: any, year: string) {

        const currentYear = new Date().getFullYear();
        const targetYear = year === 'thisyear' ? currentYear : currentYear - 1;

        const eventCount: Map<string, number> = new Map();
        for (let month = 1; month <= 12; month++) {
            const formattedMonth = `${targetYear.toString().slice(2)}/${month.toString().padStart(2, '0')}`;
            eventCount.set(formattedMonth, 0);
        }

        data.forEach(row => {
            
            const rowDate = excelSerialToJSDate(row['创建时间']);
            rowDate.setHours(rowDate.getHours() - eventMonthDistributionStrategy.TIMEDIFFERENCE);
            const rowYear = rowDate.getFullYear();
            const rowMonth = rowDate.getMonth() + 1; // getMonth() returns 0-11
            const formattedMonth = `${rowYear.toString().slice(2)}/${rowMonth.toString().padStart(2, '0')}`;

            if (rowYear == targetYear) {
                eventCount.set(formattedMonth, (eventCount.get(formattedMonth) || 0) + 1);
            }
        });

        return eventCount;
    }
}
