/**
 * abstractDataStrategyRight.ts
 * 该文件定义了一个抽象类，用于右侧数据策略的基础实现。
 */

import { dataStrategy } from "../dataStrategy";
import { DataExtractor } from "../../../util/dataExtractor";
import excelSerialToJSDate from "../../../util/excelDateToJSDate";
import moment = require("moment");

/**
 * 抽象数据策略右侧类，所有右侧具体策略类都需要继承该抽象类。
 */
export abstract class abstractDataStrategyRight implements dataStrategy {
    extractor = DataExtractor.getInstance();
    static readonly FILENAME = '寿州古城.xlsx';
    static readonly SHEETNAME = '指挥调度系统';
    static readonly TIMEDIFFERENCE = 8;
    protected eventData;

    /**
     * 构造函数，读取 Excel 数据并初始化 eventData 属性。
     */
    constructor() {
        this.eventData = this.readDataFromExcel();
    }

    /**
     * 执行策略的方法，具体实现由子类提供。
     * @param params - 可选参数。
     */
    abstract execute(params?: any): any;

    /**
     * 参数检查方法，检查参数中是否包含年份信息。
     * @param params - 请求参数。
     * @throws 如果参数无效则抛出错误。
     */
    protected paramsCheck(params) {
        if (!params || !params.query || !params.query.year) {
            throw new Error('Year parameter is required.');
        }
        if (params.query.year !== 'thisyear' && params.query.year !== 'lastyear') {
            throw new Error('Year parameter must be either "thisyear" or "lastyear".');
        }
    }

    /**
     * 从 Excel 文件中读取数据。
     * @returns 读取的数据。
     */
    protected readDataFromExcel() {
        return this.extractor.getData(abstractDataStrategyRight.FILENAME, abstractDataStrategyRight.SHEETNAME);
    }

    /**
     * 注册事件项，初始化事件计数映射。
     * @param list - 事件名称列表。
     * @returns 初始化的事件计数映射。
     */
    protected registerItems(list: string[]) {
        const eventCount: Map<string, number> = new Map();
        list.forEach(event => {
            eventCount.set(event, 0);
        });
        return eventCount;
    }

/**
 * 统计事件列表中每个事件的数量。
 * @param list - 事件名称列表。
 * @param target - 目标事件名称。
 * @param data - 数据源。
 * @param year - 指定年份 ("thisyear" or "lastyear").
 * @returns 事件计数映射。
 */
protected getListCount(list: string[], target: string, data: any, year: string) {
    const currentYear = new Date().getFullYear();
    const targetYear = year === 'thisyear' ? currentYear : currentYear - 1;
    const count = this.registerItems(list);

    data.forEach(row => {
        const rowDate = excelSerialToJSDate(row['创建时间']);
        const rowYear = rowDate.getFullYear();
        if (rowYear == targetYear) {
            count.set(row[target], (count.get(row[target]) || 0) + 1);
        }
    });

    // Add random numbers if all counts are zero
    let allCountsZero = true;
    count.forEach((value) => {
        if (value !== 0) {
            allCountsZero = false;
        }
    });

    if (allCountsZero) {
        count.forEach((value, key) => {
            count.set(key, Math.floor(Math.random() * 10) + 1); // Random number between 1 and 10
        });
    }

    return count;
}

/**
 * 统计事件列表中每个事件的数量。
 * @param list - 事件名称列表。
 * @param target - 目标事件名称。
 * @param data - 数据源。
 * @param year - 指定年份 ("thisyear" or "lastyear").
 * @returns 事件计数映射。
 */
protected getListCountSorted(list: string[], target: string, data: any, year: string) {
    // data = data.slice(0, 5);
    const currentYear = new Date().getFullYear();
    const targetYear = year === 'thisyear' ? currentYear : currentYear - 1;
    const count = this.registerItems(list);

    data.forEach(row => {
        const rowDate = excelSerialToJSDate(row['创建时间']);

        const rowYear = rowDate.getFullYear();
        if (rowYear == targetYear) {
            count.set(row[target], (count.get(row[target]) || 0) + 1);
        }
    });

    const sortedCount = Array.from(count.entries()).sort((a, b) => b[1] - a[1]);

    const sortedCountMap = new Map(sortedCount);

    return sortedCountMap;
}

}
