/**
 * allEventDataStrategy.ts
 * 该文件定义了所有事件数据策略的具体实现，获取所有事件并提取相关信息。
 */

import { abstractDataStrategyMid } from "./abstractDataStrategyMid";
import excelSerialToJSDate from "../../../util/excelDateToJSDate";

/**
 * 事件数据策略类，继承自 abstractDataStrategyMid。
 */
export class allEventDataStrategy extends abstractDataStrategyMid {
    static readonly RELEVANTINFO = ['事件标题', '处置状态', '事件类型', '事件子类型', '事件来源', '创建时间', '处置时间', '处置完成时间', '办结时间'];

    static readonly KEY_MAP = {
        '事件标题': 'eventTitle',
        '处置状态': 'dispositionStatus',
        '事件类型': 'eventType',
        '事件子类型': 'eventSubtype',
        '事件来源': 'eventSource',
        '创建时间': 'creationTime',
        '处置时间': 'dispositionTime',
        '处置完成时间': 'dispositionCompletionTime',
        '办结时间': 'closureTime'
    };

    /**
     * 执行所有事件数据策略。
     * @returns 事件的数据。
     */
    execute(): any {
        const eventData = this.extractor.getData(allEventDataStrategy.fileName, allEventDataStrategy.sheetName);
        const formattedData = this.formatDataForFrontend(eventData);
        return formattedData;
    }

    /**
     * 将键转换为英文，并格式化日期
     * @param data - 处理后的数据
     * @returns 格式化后的前端数据
     */
    private formatDataForFrontend(data: any): any {
        return data.map(row => {
            let formattedRow: { [key: string]: any } = {};

            allEventDataStrategy.RELEVANTINFO.forEach(key => {
                const translatedKey = allEventDataStrategy.KEY_MAP[key];
                let value = row[key];

                if (['创建时间', '处置时间', '处置完成时间', '办结时间'].includes(key) && value) {
                    const date = excelSerialToJSDate(value);
                    const datePart = date.toISOString().split('T')[0].replace(/-/g, '/'); 
                    const timePart = date.toTimeString().split(' ')[0].substring(0, 8);
                    value = `${datePart} ${timePart}`;
                }
                formattedRow[translatedKey] = value !== undefined ? value : null;
            });

            formattedRow.detail = this.constructDetailSection(formattedRow);

            return formattedRow;
        });
    }

    /**
     * 构建详细信息部分
     * @param row - 事件数据行
     * @returns 详细信息部分
     */
    private constructDetailSection(row: { [key: string]: any }): any {
        const details: any = [];

        if (row.creationTime) {
            details.push({
                time: row.creationTime,
                status: '待调度'
            });
        }

        if (row.dispositionTime) {
            details.push({
                time: row.dispositionTime,
                status: '处置中'
            });
        }

        if (row.dispositionCompletionTime) {
            details.push({
                time: row.dispositionCompletionTime,
                status: '已处置'
            });
        }

        if (row.closureTime) {
            details.push({
                time: row.closureTime,
                status: '已办结'
            });
        }

        return details;
    }
}
