/**
 * strategyFactory.ts
 * 该文件定义了策略工厂类，用于创建各种数据策略。
 */

import { dataStrategy } from "./dataStrategy";

// 导入具体策略类
import { sightVisitorFlowByDayStrategy } from "./left/sightVisitorFlowByDayStrategy";
import { gateStatusStrategy } from "./left/gateStatusStrategy";
import { sightVisitorFlowByHourStrategy } from "./left/sightVisitorFlowPerHourStrategy";
import { guchengLoadStrategy } from "./left/guchengLoadStrategy";
import { totalVisitorFlowStrategy } from "./left/totalVisitorFlowStrategy";
import { totalVisitorFlowByHourStrategy } from "./left/totalVisitorFlowByHourStrategy";

import { currentEventStrategy } from "./middle/currentEventStrategy";
import { totalEventCountStrategy } from "./middle/totalEventCountStrategy";

import { getEventCountByYearStrategy } from "./right/getEventCountByYearStrategy";
import { eventTimeDistributionStrategy } from "./right/eventTimeDistributionStrategy";
import { eventMonthDistributionStrategy } from "./right/eventMonthDistributionStrategy";
import { eventCategoryCountStrategy } from "./right/eventCategoryCountStrategy";
import { eventSubCategoryCountStrategy } from "./right/eventSubCategoryCountStrategy";
import { eventSrcStrategy } from "./right/eventSrcStrategy";
import { gridEventCountStrategy } from "./right/gridEventCountStrategy";
import { eventProcessingTimeStrategy } from "./right/eventProcessingTimeStrategy";
import {allEventDataStrategy} from "./middle/eventDataStrategy";

/**
 * 策略工厂类，用于创建和管理各种数据策略。
 */
export class strategyFactory {
    /**
     * 存储策略类型与其对应类的映射。
     */
    private static strategies: { [key: string]: new () => dataStrategy } = {
        'allEvents': allEventDataStrategy,
        'sightVisitorFlowByDay': sightVisitorFlowByDayStrategy,
        'gateStatus': gateStatusStrategy,
        'sightVisitorFlowPerHour': sightVisitorFlowByHourStrategy,
        'guchengLoad': guchengLoadStrategy,
        'totalVisitorFlow': totalVisitorFlowStrategy,
        'totalVisitorFlowByDay': totalVisitorFlowByHourStrategy,
        'getCurrentEventCount': currentEventStrategy,
        'totalEventCount': totalEventCountStrategy,
        'getEventCountByYear': getEventCountByYearStrategy,
        'getEventTimeDistribution': eventTimeDistributionStrategy,
        'getEventMonthDistribution': eventMonthDistributionStrategy,
        'getEventCategoryCount': eventCategoryCountStrategy,
        'getEventSubCategoryCount': eventSubCategoryCountStrategy,
        'getEventSourceCount': eventSrcStrategy,
        'getGridEventCount': gridEventCountStrategy,
        'getEventProcessingTime': eventProcessingTimeStrategy,
    };

    /**
     * 创建并返回指定类型的策略实例。
     * @param type - 策略类型的字符串标识。
     * @returns 创建的策略实例。
     * @throws 如果策略类型无效则抛出错误。
     */
    static createStrategy(type: string): dataStrategy {
        const StrategyClass = this.strategies[type];
        if (!StrategyClass) {
            throw new Error('Invalid strategy type.');
        }
        return new StrategyClass();
    }

    /**
     * 注册一个新的策略。
     * @param type - 策略类型的字符串标识。
     * @param strategy - 策略类。
     */
    static registerStrategy(type: string, strategy: new () => dataStrategy) {
        this.strategies[type] = strategy;
    }
}
