/**
 * getEventCountByYearStrategy.ts
 * 该文件定义了按年统计事件数量策略的具体实现。
 */

import {abstractDataStrategyRight} from "./abstractDataStrategyRight";
import excelSerialToJSDate from "../../../util/excelDateToJSDate";
import mapToObj from "../../../util/mapToObj";

/**
 * 按年统计事件数量策略类，继承自 abstractDataStrategyRight。
 */
export class getEventCountByYearStrategy extends abstractDataStrategyRight {

    /**
     * 执行按年统计事件数量策略。
     * @param params - 请求参数，包含年份信息。
     * @returns 事件数量统计的对象表示。
     */
    execute(params?: any): any {
        // 检查参数是否有效
        this.paramsCheck(params);

        // 获取指定年份的事件数量统计，并转换为对象
        return mapToObj(this.getEventCountForYear(this.eventData, params.query.year));
    }

    /**
     * 获取指定年份的事件数量统计。
     * @param data - 从数据源提取的数据。
     * @param year - 指定年份。
     * @returns 事件数量统计的映射。
     */
    private getEventCountForYear(data: any, year: number) {
        // 定义事件列表
        const eventList = ['事件总数', '历史遗留', '办结率'];
        const eventCount = this.registerItems(eventList);

        let finishedEventCount = 0;

        data.forEach(row => {
            const rowDate = excelSerialToJSDate(row['创建时间']);
            let rowYear;
            try {
                rowYear = rowDate.getFullYear();
            } catch (e) {
                rowYear = 0;
            }
            if (rowYear == year) {
                eventCount.set('事件总数', eventCount.get('事件总数') + 1);
                if (row['处置状态'] == '已办结') {
                    finishedEventCount++;
                }
            }
            else if (rowYear == year - 1 && row['处置状态'] != '已办结') {
                eventCount.set('历史遗留', eventCount.get('历史遗留') + 1);
            }
        });

        // 计算办结率
        eventCount.set('办结率', finishedEventCount / eventCount.get('事件总数'));
        return eventCount;
    }
}