/**
 * currentEventStrategy.ts
 * 该文件定义了当前事件策略的具体实现，获取当前日期事件的计数映射
 */

import {abstractDataStrategyMid} from "./abstractDataStrategyMid";
import excelSerialToJSDate from "../../../util/excelDateToJSDate";
import mapToObj from "../../../util/mapToObj";

/**
 * 当前事件策略类，继承自 abstractDataStrategyMid。
 */
export class currentEventStrategy extends abstractDataStrategyMid {

    /**
     * 执行事件数据策略。
     * @param params - 请求参数。
     * @returns 指定日期事件已处置和带处置事件计数
     */
    execute(params?: any): any {
        if (!params || !params.query || !params.query.date) {
            throw new Error("Date parameter is required.")
        }
        let dayData = this.extractor.getData(currentEventStrategy.fileName, currentEventStrategy.sheetName);
        return mapToObj(this.getCurrentDayEventDate(dayData, params.query.date));
    }

    /**
     * 获取当前日期事件的计数映射
     * @param data - 从数据源提取的数据。
     * @param date - 指定日期。
     * @return 指定日期事件已处置和带处置事件计数的映射
     */
    private getCurrentDayEventDate(data: any, date: string) {
        const eventCount: Map<string, number> = new Map();
        eventCount.set('pendingEvents', 0);
        eventCount.set('processedEvents', 0);

        data.forEach(row => {
            const rowDate = excelSerialToJSDate(row['数据更新时间']);
            let rowDateString;
            try {
                rowDateString = rowDate.toISOString().split('T')[0];
            }
            catch (e) {
                rowDateString = 'invalid time'
            }
            if (rowDateString === date) {
                if (row['处置状态'] == '待调度' || row['处置状态'] == '处置中') {
                    eventCount.set('pendingEvents', eventCount.get('pendingEvents') + 1);
                }
                else {
                    eventCount.set('processedEvents', eventCount.get('processedEvents') + 1);
                }
            }
        });

        return eventCount;
    }

}